<?php
namespace App\Export;

use App\Lib\SimpleXlsxWriter;
use App\Storage\Db;

/**
 * No-Composer Excel exporter.
 * Generates an .xlsx compatible with Salla import template headers.
 */
class ExcelExporter
{
    private const HEADERS = [
        'النوع ',
        'أسم المنتج',
        'تصنيف المنتج',
        'صورة المنتج',
        'وصف صورة المنتج',
        'نوع المنتج',
        'سعر المنتج',
        'الوصف',
        'هل يتطلب شحن؟',
        'رمز المنتج sku',
        'سعر التكلفة',
        'السعر المخفض',
        'تاريخ بداية التخفيض',
        'تاريخ نهاية التخفيض',
        'اقصي كمية لكل عميل',
        'إخفاء خيار تحديد الكمية',
        'اضافة صورة عند الطلب',
        'الوزن',
        'وحدة الوزن',
        'الماركة',
        'العنوان الترويجي',
        'تثبيت المنتج',
        'الباركود',
        'السعرات الحرارية',
        'MPN',
        'GTIN',
        'خاضع للضريبة ؟',
        'سبب عدم الخضوع للضريبة',
        '[1] الاسم',
        '[1] النوع',
        '[1] القيمة',
        '[1] الصورة / اللون',
        '[2] الاسم',
        '[2] النوع',
        '[2] القيمة',
        '[2] الصورة / اللون',
        '[3] الاسم',
        '[3] النوع',
        '[3] القيمة',
        '[3] الصورة / اللون',
    ];

    public function __construct(
        private Db $db,
        private string $templatePath, // kept for UI compatibility
        private string $exportsDir,
        private bool $taxEnabled = true,
        private string $imagesMode = 'first' // first|all
    ) {}

    public function export(): string
    {
        $rows = [];

        // Row 1: template section header (to resemble Salla template)
        $row1 = array_fill(0, count(self::HEADERS), '');
        $row1[0] = 'بيانات المنتج';
        $rows[] = $row1;

        // Row 2: headers
        $rows[] = self::HEADERS;

        $products = $this->db->getProducts(['status' => null]);
        $exportedIds = [];

        foreach ($products as $p) {
            $pid = (int)$p['id'];
            $details = $this->db->getProduct($pid);
            if (!$details) continue;

            $cats = [];
            if (!empty($details['categories']) && is_array($details['categories'])) {
                $cats = array_values(array_filter($details['categories'], fn($c) => $c && $c !== 'Home' && $c !== 'الرئيسية'));
            }
            $categoryPath = $cats ? implode(' > ', $cats) : '';

            $imageUrls = array_map(fn($im) => $im['image_url'], $details['images'] ?? []);
            if (empty($imageUrls) && !empty($details['main_image_url'])) $imageUrls = [$details['main_image_url']];

            if ($this->imagesMode === 'all') {
                $imageCell = implode("\n", array_slice($imageUrls, 0, 10));
            } else {
                $imageCell = $imageUrls[0] ?? '';
            }

            $desc = $details['description_html'] ?? $details['description_text'] ?? '';

            $row = array_fill(0, count(self::HEADERS), '');

            // Fill basic required columns
            $row[self::idx('النوع ')] = 'منتج';
            $row[self::idx('أسم المنتج')] = $details['name'] ?? '';
            $row[self::idx('تصنيف المنتج')] = $categoryPath;
            $row[self::idx('صورة المنتج')] = $imageCell;
            $row[self::idx('وصف صورة المنتج')] = '';
            $row[self::idx('نوع المنتج')] = 'منتج جاهز';
            $row[self::idx('سعر المنتج')] = $details['price'] ?? '';
            $row[self::idx('الوصف')] = $desc;
            $row[self::idx('هل يتطلب شحن؟')] = 'نعم';
            $row[self::idx('رمز المنتج sku')] = $details['sku'] ?? '';
            $row[self::idx('الماركة')] = $details['brand'] ?? '';

            // Tax
            $row[self::idx('خاضع للضريبة ؟')] = $this->taxEnabled ? 'نعم' : 'لا';
            $row[self::idx('سبب عدم الخضوع للضريبة')] = $this->taxEnabled ? '' : '';

            $rows[] = $row;
            $exportedIds[] = $pid;
        }

        if (!is_dir($this->exportsDir)) mkdir($this->exportsDir, 0775, true);
        $filename = 'salla_export_' . date('Y-m-d_His') . '.xlsx';
        $outPath = rtrim($this->exportsDir, '/') . '/' . $filename;

        SimpleXlsxWriter::save($rows, $outPath, 'Salla Products Template Sheet');

        $this->db->markExported($exportedIds);
        return $outPath;
    }

    private static function idx(string $header): int
    {
        static $map = null;
        if ($map === null) {
            $map = [];
            foreach (self::HEADERS as $i => $h) $map[$h] = $i;
        }
        return $map[$header] ?? 0;
    }
}
