<?php
declare(strict_types=1);

/**
 * Browser Beacon Ingest
 * Receives HTML from user's browser (bypasses anti-bot 403) via navigator.sendBeacon / fetch(no-cors)
 * POST fields: token, url, html
 */

$root = dirname(__DIR__);
$configFile = $root . '/config.php';
if (!file_exists($configFile)) {
  http_response_code(400);
  echo "Not installed";
  exit;
}
$config = require $configFile;

$token = $_POST['token'] ?? '';
$url   = $_POST['url'] ?? '';
$html  = $_POST['html'] ?? '';

if (!$token || !$url || !$html) {
  http_response_code(400);
  echo "Missing fields";
  exit;
}

$expected = $config['app']['ingest_token'] ?? '';
if (!$expected || !hash_equals($expected, $token)) {
  http_response_code(403);
  echo "Bad token";
  exit;
}

require_once $root . '/app/Storage/Db.php';
require_once $root . '/app/Utils/Logger.php';
require_once $root . '/app/Http/HttpClient.php';
require_once $root . '/app/Storage/ImageDownloader.php';
require_once $root . '/app/Parser/CategoryParser.php';
require_once $root . '/app/Parser/ProductParser.php';
require_once $root . '/app/Utils/Scraper.php';

@date_default_timezone_set($config['app']['timezone'] ?? 'Asia/Riyadh');

$db = new \App\Storage\Db($config['db']);
$logger = new \App\Utils\Logger($config['paths']['storage'] . '/logs/app.log');
$http = new \App\Http\HttpClient($config, $logger);
$img = new \App\Storage\ImageDownloader($http, $config['paths']['images']);
$scraper = new \App\Utils\Scraper($http, $db, $logger, $img);

// Save inbox copy (optional)
$inboxDir = rtrim($config['paths']['storage'], '/') . '/inbox';
@mkdir($inboxDir, 0755, true);
$fname = $inboxDir . '/' . date('Ymd_His') . '_' . substr(sha1($url),0,10) . '.html';
@file_put_contents($fname, $html);

// Try parse as product; if fails, parse as category
try {
  $pid = $scraper->scrapeProductHtml($html, $url, true);
  http_response_code(204);
  exit;
} catch (Throwable $e) {
  // fallback to category
  try {
    $count = $scraper->scrapeCategoryHtml($html, $url);
    http_response_code(204);
    exit;
  } catch (Throwable $e2) {
    $logger->error('beacon', $e2->getMessage());
    http_response_code(500);
    echo "Parse failed";
    exit;
  }
}
